/**
 * MIT License
 *
 * Copyright (c) 2021 Andrew Raffensperger
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * This is a near carbon-copy of the original source (link below) with the
 * TypeScript typings added and a few tweaks to make it ES3-compatible.
 *
 * See: https://github.com/adraffy/ens-normalize.js
 */
import { toUtf8CodePoints } from "@ethersproject/strings";
import { getData } from './include.js';
const r = getData();
import { read_member_array, read_mapped_map, read_emoji_trie } from './decoder.js';
// @TODO: This should be lazily loaded
const VALID = new Set(read_member_array(r));
const IGNORED = new Set(read_member_array(r));
const MAPPED = read_mapped_map(r);
const EMOJI_ROOT = read_emoji_trie(r);
//const NFC_CHECK = new Set(read_member_array(r, Array.from(VALID.values()).sort((a, b) => a - b)));
//const STOP = 0x2E;
const HYPHEN = 0x2D;
const UNDERSCORE = 0x5F;
function explode_cp(name) {
    return toUtf8CodePoints(name);
}
function filter_fe0f(cps) {
    return cps.filter(cp => cp != 0xFE0F);
}
export function ens_normalize_post_check(name) {
    for (let label of name.split('.')) {
        let cps = explode_cp(label);
        try {
            for (let i = cps.lastIndexOf(UNDERSCORE) - 1; i >= 0; i--) {
                if (cps[i] !== UNDERSCORE) {
                    throw new Error(`underscore only allowed at start`);
                }
            }
            if (cps.length >= 4 && cps.every(cp => cp < 0x80) && cps[2] === HYPHEN && cps[3] === HYPHEN) {
                throw new Error(`invalid label extension`);
            }
        }
        catch (err) {
            throw new Error(`Invalid label "${label}": ${err.message}`);
        }
    }
    return name;
}
export function ens_normalize(name) {
    return ens_normalize_post_check(normalize(name, filter_fe0f));
}
function normalize(name, emoji_filter) {
    let input = explode_cp(name).reverse(); // flip for pop
    let output = [];
    while (input.length) {
        let emoji = consume_emoji_reversed(input);
        if (emoji) {
            output.push(...emoji_filter(emoji));
            continue;
        }
        let cp = input.pop();
        if (VALID.has(cp)) {
            output.push(cp);
            continue;
        }
        if (IGNORED.has(cp)) {
            continue;
        }
        let cps = MAPPED[cp];
        if (cps) {
            output.push(...cps);
            continue;
        }
        throw new Error(`Disallowed codepoint: 0x${cp.toString(16).toUpperCase()}`);
    }
    return ens_normalize_post_check(nfc(String.fromCodePoint(...output)));
}
function nfc(s) {
    return s.normalize('NFC');
}
function consume_emoji_reversed(cps, eaten) {
    var _a;
    let node = EMOJI_ROOT;
    let emoji;
    let saved;
    let stack = [];
    let pos = cps.length;
    if (eaten)
        eaten.length = 0; // clear input buffer (if needed)
    while (pos) {
        let cp = cps[--pos];
        node = (_a = node.branches.find(x => x.set.has(cp))) === null || _a === void 0 ? void 0 : _a.node;
        if (!node)
            break;
        if (node.save) { // remember
            saved = cp;
        }
        else if (node.check) { // check exclusion
            if (cp === saved)
                break;
        }
        stack.push(cp);
        if (node.fe0f) {
            stack.push(0xFE0F);
            if (pos > 0 && cps[pos - 1] == 0xFE0F)
                pos--; // consume optional FE0F
        }
        if (node.valid) { // this is a valid emoji (so far)
            emoji = stack.slice(); // copy stack
            if (node.valid == 2)
                emoji.splice(1, 1); // delete FE0F at position 1 (RGI ZWJ don't follow spec!)
            if (eaten)
                eaten.push(...cps.slice(pos).reverse()); // copy input (if needed)
            cps.length = pos; // truncate
        }
    }
    return emoji;
}
//# sourceMappingURL=lib.js.map